/* Written by Yu-Fang Chen, Richard Mayr, and Chih-Duo Hong               */
/* Copyright (c) 2010                  	                                  */
/*                                                                        */
/* This program is free software; you can redistribute it and/or modify   */
/* it under the terms of the GNU General Public License as published by   */
/* the Free Software Foundation; either version 2 of the License, or      */
/* (at your option) any later version.                                    */
/*                                                                        */
/* This program is distributed in the hope that it will be useful,        */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/* GNU General Public License for more details.                           */
/*                                                                        */
/* You should have received a copy of the GNU General Public License      */
/* along with this program; if not, write to the Free Software            */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA*/

package algorithms;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import automata.FAState;
import automata.FiniteAutomaton;
import comparator.StateComparator;
import comparator.StatePairComparator;
import datastructure.HashSet;
import datastructure.Pair;
import datastructure.State_Label;


/**
 * 
 * @author Yu-Fang Chen and Chih-Duo Hong
 * 
 */
public class Simulation{
	/**
	 * Compute forward simulation relation of a Buchi automaton
	 * @param omega: a Buchi automaton
	 * @param FSim: the maximal bound of simulation relation
	 * 
	 * @return maximal simulation relation on states of the input automaton with FSim
	 */
	public Set<Pair<FAState,FAState>> FSimRelNBW(FiniteAutomaton omega, Set<Pair<FAState,FAState>> FSim) {
		Set<Pair<FAState,FAState>> nextFSim=new TreeSet<Pair<FAState,FAState>>(new StatePairComparator());		
		boolean changed = true;
		while (changed) {
			changed = false;
			Iterator<Pair<FAState,FAState>> FSim_it=FSim.iterator();
			while(FSim_it.hasNext()){
				Pair<FAState,FAState> pair=FSim_it.next();
				if (NextStateSimulated(FSim, omega, pair.getLeft(), pair.getRight())) {
					nextFSim.add(new Pair<FAState, FAState>(pair.getLeft(),pair.getRight()));
				}else{
					changed=true;
				}
			}

			FSim=nextFSim;
			nextFSim=new TreeSet<Pair<FAState,FAState>>(new StatePairComparator());
		}
		return FSim;
	}


	public Set<Pair<FAState,FAState>> FastFSimRelNBW(FiniteAutomaton omega, boolean[][] fsim) 
	{
		return FastFSimRelNBW(omega,null,fsim);
	}	
	
	public Set<Pair<FAState,FAState>> FastFSimRelNBW(FiniteAutomaton omega1,FiniteAutomaton omega2, boolean[][] fsim) 
	{


		ArrayList<FAState> all_states=new ArrayList<FAState>();
		HashSet<String> alphabet=new HashSet<String>();
		
		all_states.addAll(omega1.states);
		alphabet.addAll(omega1.alphabet);

		if(omega2!=null){
			all_states.addAll(omega2.states);
			alphabet.addAll(omega2.alphabet);
		}
		//implement the HHK algorithm
		int n_states = all_states.size();
		int n_symbols = alphabet.size();
		FAState[] states = all_states.toArray(new FAState[0]);
		ArrayList<String> symbols=new ArrayList<String>(alphabet);
		

		// fsim[u][v]=true iff v in fsim(u) iff v forward-simulates u
		
		int[][][] pre = new int[n_symbols][n_states][];
		int[][][] post = new int[n_symbols][n_states][];
		int[][] pre_len = new int[n_symbols][n_states];
		int[][] post_len = new int[n_symbols][n_states];

		    // Initialize memory of pre/post
		for(int s=0;s<n_symbols;s++)
		{
			String a = symbols.get(s);
			    for(int p=0; p<n_states; p++){
				Set<FAState> next = states[p].getNext(a);
				post_len[s][p]=0;
				if (next != null) post[s][p] = new int[states[p].getNext(a).size()];
				Set<FAState> prev = states[p].getPre(a);
				pre_len[s][p]=0;
				if (prev != null) pre[s][p] = new int[states[p].getPre(a).size()];
			    }
		}

		//state[post[s][q][r]] is in post_s(q) for 0<=r<adj_len[s][q]
		//state[pre[s][q][r]] is in pre_s(q) for 0<=r<adj_len[s][q]
		for(int s=0;s<n_symbols;s++)
		{
			String a = symbols.get(s);
			    for(int p=0; p<n_states; p++){
				Set<FAState> next = states[p].getNext(a);
				if (next != null){
				for(int q=0; q<n_states; q++)		
				{
					if(next.contains(states[q]))
					{
						//if p --a--> q, then p is in pre_a(q), q is in post_a(p) 
						pre[s][q][pre_len[s][q]++] = p;
						post[s][p][post_len[s][p]++] = q;
					}
				}
				}
			    }
		}

		int[] todo = new int[n_states*n_symbols];
		int todo_len = 0;
		
		int[][][] remove = new int[n_symbols][n_states][n_states];
		int[][] remove_len = new int[n_symbols][n_states];
		for(int a=0; a<n_symbols; a++)
		{
			for(int p=0; p<n_states; p++)
				if(pre_len[a][p]>0) // p is in a_S
				{	
					Sharpen_S_a:
					for(int q=0; q<n_states; q++)	// {all q} --> S_a 
					{
							if(post_len[a][q]>0)	/// q is in S_a 
							{	
								for(int r=0; r<post_len[a][q]; r++) 
									if(fsim[p][post[a][q][r]]) 	// q is in pre_a(sim(p))
										continue Sharpen_S_a;	// skip q						
								remove[a][p][remove_len[a][p]++] = q;
							}
					}
					if(remove_len[a][p]>0)
						todo[todo_len++] = a*n_states + p;
				}
		}
		int[] swap = new int[n_states];
		int swap_len = 0;
		boolean using_swap = false;
		
		while(todo_len>0)
		{
			todo_len--;
			int v = todo[todo_len] % n_states;
			int a = todo[todo_len] / n_states;
			int len = (using_swap? swap_len : remove_len[a][v]);
			remove_len[a][v] = 0;
			
			for(int j=0; j<pre_len[a][v]; j++)
			{
				int u = pre[a][v][j];
				
				for(int i=0; i<len; i++)			
				{
					int w = (using_swap? swap[i] : remove[a][v][i]);
					if(fsim[u][w]) 
					{
						fsim[u][w] = false;					
						for(int b=0; b<n_symbols; b++)
							if(pre_len[b][u]>0)
							{
								Sharpen_pre_b_w:
								for(int k=0; k<pre_len[b][w]; k++)
								{	
									int ww = pre[b][w][k];
									for(int r=0; r<post_len[b][ww]; r++) 
										if(fsim[u][post[b][ww][r]]) 	// ww is in pre_b(sim(u))
											continue Sharpen_pre_b_w;	// skip ww
									
									if(b==a && u==v && !using_swap)
										swap[swap_len++] = ww;
									else{										
										if(remove_len[b][u]==0)
											todo[todo_len++] = b*n_states + u;
										remove[b][u][remove_len[b][u]++] = ww;
									}
									
								}
							}
					}//End of if(fsim[u][w])
				}				
			}			
			if(swap_len>0)
			{	
				if(!using_swap)
				{	
					todo[todo_len++] = a*n_states + v;	
					using_swap = true; 
				}else{
					swap_len = 0;
					using_swap = false;
				}
			}
			
		}

		Set<Pair<FAState,FAState>> FSim2 = new TreeSet<Pair<FAState,FAState>>(new StatePairComparator());
		for(int p=0; p<n_states; p++)	
			for(int q=0; q<n_states; q++)
				if(fsim[p][q]) // q is in sim(p), q simulates p
					FSim2.add(new Pair<FAState, FAState>(states[p],states[q]));
		return FSim2;
		
	}	

	public Set<Pair<FAState,FAState>> FastBSimRelNBW(FiniteAutomaton omega1,FiniteAutomaton omega2, boolean[][] bsim) 
	{


		ArrayList<FAState> all_states=new ArrayList<FAState>();
		HashSet<String> alphabet=new HashSet<String>();
		
		all_states.addAll(omega1.states);
		alphabet.addAll(omega1.alphabet);

		if(omega2!=null){
			all_states.addAll(omega2.states);
			alphabet.addAll(omega2.alphabet);
		}
		//implement the HHK algorithm
		int n_states = all_states.size();
		int n_symbols = alphabet.size();
		FAState[] states = all_states.toArray(new FAState[0]);
		ArrayList<String> symbols=new ArrayList<String>(alphabet);
		

		// fsim[u][v]=true iff v in fsim(u) iff v forward-simulates u
		
		int[][][] pre = new int[n_symbols][n_states][];
		int[][][] post = new int[n_symbols][n_states][];
		int[][] pre_len = new int[n_symbols][n_states];
		int[][] post_len = new int[n_symbols][n_states];
		
		    // Initialize memory of pre/post. Pre/Post reversed because of bw-sim.
		for(int s=0;s<n_symbols;s++)
		{
			String a = symbols.get(s);
			    for(int p=0; p<n_states; p++){
				post_len[s][p]=0;
				pre_len[s][p]=0;
				Set<FAState> next = states[p].getNext(a);
				if (next != null) pre[s][p] = new int[states[p].getNext(a).size()];
				Set<FAState> prev = states[p].getPre(a);
				if (prev != null) post[s][p] = new int[states[p].getPre(a).size()];
			    }
		}

		//state[post[s][q][r]] is in post_s(q) for 0<=r<adj_len[s][q]
		//state[pre[s][q][r]] is in pre_s(q) for 0<=r<adj_len[s][q]
		for(int s=0;s<n_symbols;s++)
		{
			String a = symbols.get(s);
			for(int p=0; p<n_states; p++)
				for(int q=0; q<n_states; q++)		
				{
					Set<FAState> prev = states[p].getPre(a); 
					if(prev!=null && prev.contains(states[q]))
					{
						//if p --a--> q, then p is in pre_a(q), q is in post_a(p) (note: it is backward)
						pre[s][q][pre_len[s][q]++] = p;
						post[s][p][post_len[s][p]++] = q;
					}
				}
		}
		int[] todo = new int[n_states*n_symbols];
		int todo_len = 0;
		
		int[][][] remove = new int[n_symbols][n_states][n_states];
		int[][] remove_len = new int[n_symbols][n_states];
		for(int a=0; a<n_symbols; a++)
		{
			for(int p=0; p<n_states; p++)
				if(pre_len[a][p]>0) // p is in a_S
				{	
					Sharpen_S_a:
					for(int q=0; q<n_states; q++)	// {all q} --> S_a 
					{
							if(post_len[a][q]>0)	/// q is in S_a 
							{	
								for(int r=0; r<post_len[a][q]; r++) 
									if(bsim[p][post[a][q][r]]) 	// q is in pre_a(sim(p))
										continue Sharpen_S_a;	// skip q						
								remove[a][p][remove_len[a][p]++] = q;
							}
					}
					if(remove_len[a][p]>0)
						todo[todo_len++] = a*n_states + p;
				}
		}
		int[] swap = new int[n_states];
		int swap_len = 0;
		boolean using_swap = false;
		
		while(todo_len>0)
		{
			todo_len--;
			int v = todo[todo_len] % n_states;
			int a = todo[todo_len] / n_states;
			int len = (using_swap? swap_len : remove_len[a][v]);
			remove_len[a][v] = 0;
			
			for(int j=0; j<pre_len[a][v]; j++)
			{
				int u = pre[a][v][j];
				
				for(int i=0; i<len; i++)			
				{
					int w = (using_swap? swap[i] : remove[a][v][i]);
					if(bsim[u][w]) 
					{
						bsim[u][w] = false;					
						for(int b=0; b<n_symbols; b++)
							if(pre_len[b][u]>0)
							{
								Sharpen_pre_b_w:
								for(int k=0; k<pre_len[b][w]; k++)
								{	
									int ww = pre[b][w][k];
									for(int r=0; r<post_len[b][ww]; r++) 
										if(bsim[u][post[b][ww][r]]) 	// ww is in pre_b(sim(u))
											continue Sharpen_pre_b_w;	// skip ww
									
									if(b==a && u==v && !using_swap)
										swap[swap_len++] = ww;
									else{										
										if(remove_len[b][u]==0)
											todo[todo_len++] = b*n_states + u;
										remove[b][u][remove_len[b][u]++] = ww;
									}
									
								}
							}
					}//End of if(fsim[u][w])
				}				
			}			
			if(swap_len>0)
			{	
				if(!using_swap)
				{	
					todo[todo_len++] = a*n_states + v;	
					using_swap = true; 
				}else{
					swap_len = 0;
					using_swap = false;
				}
			}
			
		}

		Set<Pair<FAState,FAState>> BSim2 = new TreeSet<Pair<FAState,FAState>>(new StatePairComparator());
		for(int p=0; p<n_states; p++)	
			for(int q=0; q<n_states; q++)
				if(bsim[p][q]) // q is in sim(p), q simulates p
					BSim2.add(new Pair<FAState, FAState>(states[p],states[q]));
		return BSim2;
		
	}	
	
	
	
	/**
	 * Compute forward simulation relation of a Buchi automaton using Henzinger, Henzinger, Kopke FOCS 1995
	 * @param omega: a Buchi automaton
	 * @param FSim: maximum simulation relation
	 * 
	 * @return simulation relation on states of the input automaton
	 */
	public Set<Pair<FAState,FAState>> FastFSimRelNBW2(FiniteAutomaton omega, Set<Pair<FAState,FAState>> FSim) {
		TreeMap<State_Label, Set<FAState>> Remove=new TreeMap<State_Label, Set<FAState>>();

		HashMap<String,Integer> symMap=new HashMap<String,Integer>();
		int [][][] counter = new int[omega.states.size()][omega.states.size()][omega.alphabet.size()];
		for(int i=0;i<omega.states.size();i++){
			for(int j=0;j<omega.states.size();j++){
				for(int k=0;k<omega.alphabet.size();k++){
					counter[i][j][k]=0;
				}
			}
		}
		
		Iterator<FAState> state_it=omega.states.iterator();
		while(state_it.hasNext()){
		FAState v=state_it.next();
			Iterator<String> sym_it=omega.getAllTransitionSymbols().iterator();
			int sym_index=0;
			while(sym_it.hasNext()){
				String sym=sym_it.next();
				symMap.put(sym, sym_index);
				sym_index++;
				Set<FAState> allStates=new TreeSet<FAState>();
				allStates.addAll(omega.states);
				Remove.put(new State_Label(v,sym), allStates);
			}
		}
		Iterator<Pair<FAState,FAState>> FSim_it=FSim.iterator();

		while(FSim_it.hasNext()){
			Pair<FAState,FAState> cur=FSim_it.next();
			FAState v=cur.getLeft();
			FAState sim_v=cur.getRight();
			
			Iterator<String> symbol_it=sim_v.preIt();
			while(symbol_it.hasNext()){
				String symbol=symbol_it.next();

				Iterator<FAState> from_it=sim_v.getPre(symbol).iterator();
				while(from_it.hasNext()){
					FAState from=from_it.next();
					if(Remove.get(new State_Label(v,symbol))!=null)
						Remove.get(new State_Label(v,symbol)).remove(from);
					counter[from.id][v.id][symMap.get(symbol)]++;
				}
			}
		}

		while(!Remove.isEmpty()){
			State_Label key=Remove.keySet().iterator().next();
			Set<FAState> remove=Remove.get(key);
			Remove.remove(key);
			FAState v=key.getState();
			String symbol=key.getLabel();
			if(v.getPre(symbol)==null)
				continue;			
			
			Iterator<FAState> pre_it=v.getPre(symbol).iterator();
			while(pre_it.hasNext()){
				FAState u=pre_it.next();
				Iterator<FAState> remove_it=remove.iterator();
				while(remove_it.hasNext()){
					FAState w=remove_it.next();
					if(FSim.contains(new Pair<FAState,FAState>(u,w))){
						FSim.remove(new Pair<FAState,FAState>(u,w));

						Iterator<String> symbol_it=w.preIt();
						while(symbol_it.hasNext()){
							String w_symbol=symbol_it.next();

							Iterator<FAState> w_pre_it=w.getPre(w_symbol).iterator();
							while(w_pre_it.hasNext()){
								FAState w_pre=w_pre_it.next();
								counter[w_pre.id][u.id][symMap.get(w_symbol)]--;
								if(counter[w_pre.id][u.id][symMap.get(w_symbol)]==0){
									if(!Remove.containsKey(new State_Label(u,w_symbol))){
										Set<FAState> emptyStates=new TreeSet<FAState>(new StateComparator());
										Remove.put(new State_Label(u,w_symbol), emptyStates);
									}
									Remove.get(new State_Label(u,w_symbol)).add(w_pre);
								}
							}
						}
					}
				}
			}
		}
		return FSim;
	}

	
	/**
	 * Compute backward simulation relation of a Buchi automaton
	 * @param omega: a Buchi automaton
	 * @param BSim: the maximal bound of simulation relation
	 * 
	 * @return maximal simulation relation on states of the input automaton with BSim
	 */
	public Set<Pair<FAState,FAState>> BSimRelNBW(FiniteAutomaton omega, Set<Pair<FAState,FAState>> BSim) {
		Set<Pair<FAState,FAState>> nextBSim=new TreeSet<Pair<FAState,FAState>>(new StatePairComparator());		
		boolean changed = true;
		while (changed) {
			changed = false;
			Iterator<Pair<FAState,FAState>> BSim_it=BSim.iterator();
			while(BSim_it.hasNext()){
				Pair<FAState,FAState> pair=BSim_it.next();
				if (PreStateSimulated(BSim, omega, pair.getLeft(), pair.getRight())) {
					nextBSim.add(new Pair<FAState, FAState>(pair.getLeft(),pair.getRight()));
				}else{
					changed=true;
				}
			}

			BSim=nextBSim;
			nextBSim=new TreeSet<Pair<FAState,FAState>>(new StatePairComparator());
		}
		return BSim;
	}
	
	private boolean NextStateSimulated(Set<Pair<FAState, FAState>> sim,
			FiniteAutomaton omega, FAState p, FAState q) {
		Iterator<String> symbol_it=p.nextIt();
		while(symbol_it.hasNext()){
			String a=symbol_it.next();
			Iterator<FAState> p_states=p.getNext(a).iterator();
			if(q.getNext(a)==null)
				return false;
			while(p_states.hasNext()){
				FAState p_next=p_states.next();
				boolean hasSimulatingState = false;
				Iterator<FAState> q_states=q.getNext(a).iterator();
				while(q_states.hasNext()){
					FAState q_next=q_states.next();
					if (sim.contains(new Pair<FAState, FAState>(p_next,q_next))){
						hasSimulatingState = true;
						break;
					}
				}
				if (!hasSimulatingState) {
					return false;
				}
			}
		}
		return true;
	}	

	private boolean PreStateSimulated(Set<Pair<FAState, FAState>> sim,
			FiniteAutomaton omega, FAState p, FAState q) {
		Iterator<String> symbol_it=p.preIt();
		while(symbol_it.hasNext()){
			String a=symbol_it.next();
			Iterator<FAState> p_states=p.getPre(a).iterator();
			if(q.getPre(a)==null)
				return false;
			while(p_states.hasNext()){
				FAState p_pre=p_states.next();
				boolean hasSimulatingState = false;
				Iterator<FAState> q_states=q.getPre(a).iterator();
				while(q_states.hasNext()){
					FAState q_pre=q_states.next();
					if (sim.contains(new Pair<FAState, FAState>(p_pre,q_pre))){
						hasSimulatingState = true;
						break;
					}
				}
				if (!hasSimulatingState) {
					return false;
				}
			}
		}
		return true;
	}


	/**
	 * Compute delayed (forward) simulation relation on/between two Buchi automata
	 * @param omega1, omega2: two Buchi automata
	 *
	 * @return maximal delayed simulation relation
	 */


	// Aux. code for for delayed simulation

	private boolean trapped(int s, int q, int a, int[][][] post, int[][] post_len, boolean[][] X)
        {
	    if(post_len[a][q]>0){
		    for(int r=0; r<post_len[a][q]; r++)
			if(!X[s][post[a][q][r]]) return false;
	    }
	    return true;
	}

        private boolean CPre(int p, int q, int n_symbols, int[][][] post, int[][] post_len, boolean[][] X)
        {
	    for(int a=0; a<n_symbols; a++)
		if(post_len[a][p]>0){
		    for(int r=0; r<post_len[a][p]; r++) 
			if(trapped(post[a][p][r], q, a, post, post_len, X)) return true;
	    }
	    return false;
	}

    private void get_avoid(boolean[][] avoid, boolean[] isFinal, int n_states, int n_symbols, int[][][] post, int[][] post_len, boolean[][] W)
        {
	    //System.out.println("Computing getavoid.");
	   for(int p=0; p<n_states; p++)
		    for(int q=0; q<n_states; q++)
				    avoid[p][q]=true;
				    
		boolean changed=true;
		while(changed){
		    changed=false;
		    //System.out.println("Computing getavoid: Iterating refinement");
		    for(int p=0; p<n_states; p++)
			for(int q=0; q<n_states; q++){
			    if(W[p][q] || !avoid[p][q]) continue; // If W then it stays true. If avoid false then stay false
			    if(isFinal[q]) { avoid[p][q]=false; changed=true; continue; }
			    if(!CPre(p,q,n_symbols,post,post_len,avoid)) { avoid[p][q]=false; changed=true; }
			}
		} 

	}
	    
	public Set<Pair<FAState,FAState>> DelayedSimRelNBW(FiniteAutomaton omega1,FiniteAutomaton omega2) 
	{
		ArrayList<FAState> all_states=new ArrayList<FAState>();
		HashSet<String> alphabet=new HashSet<String>();

		all_states.addAll(omega1.states);
		alphabet.addAll(omega1.alphabet);

		if(omega2!=null){
			all_states.addAll(omega2.states);
			alphabet.addAll(omega2.alphabet);
		}

		int n_states = all_states.size();
		int n_symbols = alphabet.size();

		boolean[][] W = new boolean[n_states][n_states];
		for(int p=0; p<n_states; p++)
		    for(int q=0; q<n_states; q++)
				    W[p][q]=false;

		FAState[] states = all_states.toArray(new FAState[0]);
		{
		ArrayList<String> symbols=new ArrayList<String>(alphabet);

		boolean[] isFinal = new boolean[n_states];
		for(int i=0;i<n_states;i++){			
			isFinal[i] = states[i].getowner().F.contains(states[i]);
		}
		
		int[][][] post = new int[n_symbols][n_states][];
		int[][] post_len = new int[n_symbols][n_states];
		
		//state[post[s][q][r]] is in post_s(q) for 0<=r<adj_len[s][q]
		//state[pre[s][q][r]] is in pre_s(q) for 0<=r<adj_len[s][q]
		// System.out.println("Delayed sim: Getting post");
		for(int s=0;s<n_symbols;s++)
		{
			String a = symbols.get(s);
			for(int p=0; p<n_states; p++)
			    {
				//System.out.println("Delayed sim: Getting post: Iterating p: "+p+" of "+n_states+" s is "+s+" of "+n_symbols);
				post_len[s][p]=0;
				Set<FAState> next = states[p].getNext(a); 
				if (next != null){
				    post[s][p] = new int[states[p].getNext(a).size()];
				    for(int q=0; q<n_states; q++)
					{
					    if(next.contains(states[q]))
						{
						//if p --a--> q, then p is in pre_a(q), q is in post_a(p) 
						// pre[s][q][pre_len[s][q]++] = p;
						post[s][p][post_len[s][p]++] = q;
						}
					}
				}
			    }
		}
		
		// Initialize result. This will grow by least fixpoint iteration.
		boolean[][] avoid = new boolean[n_states][n_states];
				    
		boolean changed=true;
		while(changed){
		    changed=false;
		    get_avoid(avoid,isFinal,n_states,n_symbols,post,post_len,W);
		    for(int p=0; p<n_states; p++)
			for(int q=0; q<n_states; q++){
			    if(W[p][q]) continue;
			    if(isFinal[p] && avoid[p][q]) { W[p][q]=true; changed=true; continue; }
			    if(CPre(p,q,n_symbols,post,post_len,W)) { W[p][q]=true; changed=true; }
			}
		}
		}
		// Create final result as set of pairs of states
		Set<Pair<FAState,FAState>> FSim2 = new TreeSet<Pair<FAState,FAState>>(new StatePairComparator());
		for(int p=0; p<n_states; p++)	
			for(int q=0; q<n_states; q++)
				if(!W[p][q]) // W is winning for spoiler here, so the result is the opposite.
					FSim2.add(new Pair<FAState, FAState>(states[p],states[q]));
		return FSim2;
	}





	/**
	 * Compute mediated preorder relation on a Buchi automaton
	 * (mediated preorder between two automata is not meaningful).
	 * @param fa: a Buchi automaton
	 * frel, breal: correct forward/backward simulation relations on fa
	 *
	 * @return mediated preorder relation
	 * Note: Currently this relation is not used. It seems to be obsolete against -qr and -sp
	 */


public Set<Pair<FAState,FAState>> MediatedRelNBW(FiniteAutomaton fa, Set<Pair<FAState, FAState>> frel,Set<Pair<FAState, FAState>> brel)
	{
		ArrayList<FAState> all_states=new ArrayList<FAState>();
		all_states.addAll(fa.states);

		int n_states = all_states.size();
		FAState[] states = all_states.toArray(new FAState[0]);
		boolean[][] F = new boolean[n_states][n_states];
		boolean[][] B = new boolean[n_states][n_states];
		boolean[][] M = new boolean[n_states][n_states];
		for(int p=0; p<n_states; p++)
		    for(int q=0; q<n_states; q++)
			{
			    F[p][q] = frel.contains(new Pair<FAState, FAState>(states[p], states[q]));
			    B[p][q] = brel.contains(new Pair<FAState, FAState>(states[p], states[q]));
			    M[p][q] = false;
			}
		// Compute M = F B^-1
		for(int p=0; p<n_states; p++)
		    for(int q=0; q<n_states; q++)
			if(F[p][q])
			    {			
				for(int r=0; r<n_states; r++) 
				    if(B[r][q]) M[p][r]=true;
			    }
		
		// Now refine s.t. MF is contained in M
		boolean changed=true;
		while(changed){
		    changed=false;
		    for(int p=0; p<n_states; p++)
			for(int q=0; q<n_states; q++){
			    if(!M[p][q]) continue;
			    for(int r=0; r<n_states; r++) 
				{
				    // M[p][q] is true here
				    if(F[q][r] && !M[p][r]) { M[p][q]=false; changed=true; break; }
				}
			}
		}
		// Create final result as set of pairs of states
		Set<Pair<FAState,FAState>> FSim2 = new TreeSet<Pair<FAState,FAState>>(new StatePairComparator());
		for(int p=0; p<n_states; p++)	
			for(int q=0; q<n_states; q++)
			    if(M[p][q]) FSim2.add(new Pair<FAState, FAState>(states[p],states[q]));
		return FSim2;
}

}


